 let autoRefreshInterval = null; // Store the interval reference for auto-refresh

document.addEventListener('DOMContentLoaded', function () {
    const refreshToggle = document.getElementById('refresh-toggle');
    const manualRefreshBtn = document.getElementById('manual-refresh-btn');
    const startChronoBtn = document.getElementById('start-chrono-btn');
    const stopChronoBtn = document.getElementById('stop-chrono-btn');
    let countdownInterval = null;

    fetchAndUpdateContent(); 

    function stopAutoRefresh() {
        clearInterval(autoRefreshInterval);
        autoRefreshInterval = null;
    }

    refreshToggle.addEventListener('change', function () {
        if (refreshToggle.checked) {
            autoRefreshInterval = setInterval(fetchAndUpdateContent, 5000);
        } else {
            stopAutoRefresh();
        }
    });

    manualRefreshBtn.addEventListener('click', fetchAndUpdateContent);

    startChronoBtn.addEventListener('click', function () {
        startChrono();
    });

    stopChronoBtn.addEventListener('click', function () {
        stopChrono();
    });

     // Function to stop any existing countdown and clear interval
    function clearCountdownInterval() {
        if (countdownInterval) {
            clearInterval(countdownInterval);  // Clear the existing interval
            countdownInterval = null;          // Reset the variable
        }
    }
    
    // Function to start the chrono (set targettime to now + 20 minutes)
    function startChrono(targetTime = null) {
        if (!targetTime) {
            // Fetch new targettime from the server (now + 20 minutes)
            fetch('start_chrono.php')
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        clearCountdownInterval();  // Clear any existing countdown
                        updateRemainingTime(data.targettime); // Start the countdown with the new targettime
                        console.log("success");
                    } else {
                        console.error('Error starting chrono:', data.message);
                    }
                })
                .catch(error => console.error('Error starting chrono:', error));
        } else {
            // If targetTime is provided, use it (e.g., stop countdown with an old date)
            //clearCountdownInterval();
            //updateRemainingTime(targetTime); // Use the provided targetTime
        }
    }

    // Function to stop the chrono (use a past date to stop countdown)
    function stopChrono() {
         fetch('stop_chrono.php')
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        clearCountdownInterval();  // Clear any existing countdown
                        //updateRemainingTime(data.targettime); // Start the countdown with the new targettime
                        
                    } else {
                        console.error('Error starting chrono:', data.message);
                    }
                })
                .catch(error => console.error('Error starting chrono:', error));
        const pastDate = "1970-01-01T00:00:00Z"; // An obsolete past date
        
        clearCountdownInterval();
        document.getElementById('remaining-time').innerText = "00:00";
        //startChrono(pastDate);  // Pass the past date to stop the countdown properly
    }

    // Function to update the remaining time for the countdown
    function updateRemainingTime(targetTimeSRC) {
        // Clear any existing countdown to avoid multiple intervals
        clearCountdownInterval();

        // Start a new countdown interval
        countdownInterval = setInterval(function () {
            const targetTime = new Date(targetTimeSRC); // Convert the target time from string to Date
            const now = new Date(); // Current time
            const diff = targetTime - now; // Calculate the remaining time in milliseconds

            if (diff <= 0) {
                document.getElementById('remaining-time').innerText = "00:00";
                clearCountdownInterval(); // Stop the interval when time is up
                return;
            }

            // Convert the difference to hours, minutes, and seconds
            const hours = Math.floor(diff / (1000 * 60 * 60));
            const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((diff % (1000 * 60)) / 1000);

            // Format the time as HH:MM:SS
            const formattedTime = 
                // (hours < 10 ? "0" : "") + hours + ":" +
                (minutes < 10 ? "0" : "") + minutes + ":" +
                (seconds < 10 ? "0" : "") + seconds;

            // Update the remaining time on the page
            document.getElementById('remaining-time').innerText = formattedTime;
        }, 1000);
    }

    
    
    function fetchAndUpdateContent() {
        fetch('fetch_data.php')
        .then(response => response.json())
        .then(response => {
            const maxVagues = 15;
            const data = response.data;
            const headerData = response.header;
            const tableData = prepareTableData(data, maxVagues);
            renderHeader(headerData);
            renderCompetitorInfo(tableData);
            renderScoreTable(tableData, maxVagues);
            bindContextMenu();
            updateRemainingTime(headerData.targettime);
            console.log(headerData.targettime);
        })
        .catch(error => console.error('Error fetching data:', error));
    }

  function renderScoreTable(competitors, maxVague = 15) {
    const tableBody = document.querySelector('#scores-table tbody');
    const tableHead = document.querySelector('#scores-table thead');
    let tableHtml = '';
    const maxVagues = 15;
    
    // Generate the wave headers (W1, W2, W3...) with exactly 15 columns
    let waveHeaderHtml = '<tr><th></th>'; // First empty cell for judge column
    for (let i = 1; i <= maxVagues; i++) {
        waveHeaderHtml += `<th>W${i}</th>`;
    }
    waveHeaderHtml += '</tr>';

    // Insert the wave header row into the table header
    tableHead.innerHTML = waveHeaderHtml;

    // Render each competitor's scores
    Object.keys(competitors).forEach(ligne => {
        let competitor = competitors[ligne];

        // Render each judge's scores with color applied to empty cells
        Object.keys(competitor.judges).forEach(judge => {
            let rowHtml = `<tr><td>J${judge}</td>`; // First column for the judge
            competitor.judges[judge].forEach((wave, index) => {
                const colorClass = wave.coul || `coul-${competitor.ligne}`; // Get the color class or use default color

                // Ensure serie and concu are included
                const serie = competitor.serie;
                const concu = competitor.concu;

                if (wave.note) {
                    // Make the cell editable and include serie and concu
                    rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${index + 1}">${wave.note}</td>`;
                } else {
                    rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${index + 1}">-</td>`; // Editable empty cells as well
                }
            });

            // Fill remaining empty wave columns up to 15 if needed
            for (let i = competitor.judges[judge].length; i < maxVagues; i++) {
                const colorClass = `coul-${competitor.ligne}`;
                const serie = competitor.serie;
                const concu = competitor.concu;

                rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${i + 1}">-</td>`;
            }

            rowHtml += '</tr>';
            tableHtml += rowHtml;
        });

        // Render averages row with best scores highlighted in red and bold
        // Now pass interferenceCount and hasPriority to renderAverageRow
        tableHtml += renderAverageRow(
            competitor.averages,
            competitor.bestTwo,
            maxVagues,
            competitor.serie,
            competitor.concu,
            competitor.interferenceFlags,
            competitor.hasPriority,
            competitor.interferenceCount
        );
    });

    // Insert the table HTML into the page
    tableBody.innerHTML = tableHtml;

    // Add event listener for editing notes (if needed)
    addEditableListeners();
}


    

    function bindContextMenu() {
        const averageCells = document.querySelectorAll('.average-cell');
        averageCells.forEach(cell => {
            cell.addEventListener('click', (event) => {
                event.preventDefault();
                showContextMenu(event, cell);
            });
        });
    }

    function showContextMenu(event, cell) {
    const contextMenu = document.getElementById('context-menu');
    const x = event.pageX;
    const y = event.pageY;

    contextMenu.style.left = `${x}px`;
    contextMenu.style.top = `${y}px`;
    contextMenu.style.display = 'block';

    const serie = cell.getAttribute('data-serie');
    const concu = cell.getAttribute('data-concu');
    const wave = cell.getAttribute('data-wave');

    // Remove any existing event listeners to prevent duplicate execution
    removeExistingListeners();

    // Add new event listeners
    document.getElementById('cancel').addEventListener('click', hideContextMenu);
    document.getElementById('remove-interference').addEventListener('click', () => updateDatabase('REMOVE INTERFERENCE', serie, concu, wave));
    document.getElementById('interference-in').addEventListener('click', () => updateDatabase('INTERFERENCE IN', serie, concu, wave));
    document.getElementById('priority-in').addEventListener('click', () => updateDatabase('PRIORITY IN', serie, concu, wave));
}

function hideContextMenu() {
    const contextMenu = document.getElementById('context-menu');
    contextMenu.style.display = 'none';
}

function updateDatabase(action, serie, concu, wave) {
    const data = { action, serie, concu, wave };

    fetch('update_action.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(data),
    })
    .then(response => {
        if (!response.ok) throw new Error(response.statusText);
        return response.json();
    })
    .then(result => {
        if (result.status === 'success') {
            console.log(`${action} updated successfully.`);
        } else {
            console.error('Error:', result.message);
        }
    })
    .catch(error => console.error('Error:', error));

    hideContextMenu(); // Hide the menu after an action is taken
}

// Function to remove existing event listeners to prevent duplicate actions
function removeExistingListeners() {
    const cancelBtn = document.getElementById('cancel');
    const removeInterferenceBtn = document.getElementById('remove-interference');
    const interferenceInBtn = document.getElementById('interference-in');
    const priorityInBtn = document.getElementById('priority-in');

    cancelBtn.replaceWith(cancelBtn.cloneNode(true));
    removeInterferenceBtn.replaceWith(removeInterferenceBtn.cloneNode(true));
    interferenceInBtn.replaceWith(interferenceInBtn.cloneNode(true));
    priorityInBtn.replaceWith(priorityInBtn.cloneNode(true));
}


    document.addEventListener('click', function (event) {
        const contextMenu = document.getElementById('context-menu');
        if (!event.target.closest('.average-cell') && !event.target.closest('#context-menu')) {
            contextMenu.style.display = 'none';
        }
    });

    
});



// Function to render the header with competition details dynamically
    function renderHeader(headerData) {
         document.getElementById('division-info').textContent = headerData.division;
        document.getElementById('round-info').textContent = headerData.tour;
        document.getElementById('heat-info').textContent = headerData.heat;
    }


// Function to prepare table data and calculate best averages
function prepareTableData(data, maxVagues) {
    const competitors = {};

    data.forEach(entry => {
        const ligne = parseInt(entry.ligne);  // Ensure 'ligne' is an integer
        const juge = parseInt(entry.juge);    // Convert 'juge' to an integer
        const vague = parseInt(entry.vague);  // Convert 'vague' to an integer
        const concu = parseInt(entry.concu);  // Convert 'concu' to an integer
        const note = parseFloat(entry.note);

        // Initialize competitor if not done already
        if (!competitors[ligne]) {
            competitors[ligne] = {
                nom: entry.nom,
                club: entry.club,
                ligne: ligne,
                serie: entry.serie,
                concu: concu,
                judges: {},
                averages: [],
                bestTwo: [0, 0],
                total: 0,
                interferenceFlags: Array(maxVagues).fill(false),
                hasPriority: false,  // Priority flag for note = 14
                interferenceCount: 0 // Interference count for note = 11
            };
        }

        // Ignore notes with juge = 0 (interference) for display
        if (juge === 0 && (note === 11 || note === 14)) {
            competitors[ligne].interferenceFlags[vague - 1] = true;

            // Mark the competitor as having priority if note = 14
            if (note === 14) {
                competitors[ligne].hasPriority = true;
            }

            // Track interference count for note = 11
            if (note === 11) {
                competitors[ligne].interferenceCount++;
            }

            return; // Skip further processing for interference
        }

        if (!competitors[ligne].judges[juge]) {
            competitors[ligne].judges[juge] = Array(maxVagues).fill('-');
        }

        // Store the note for the correct wave
        competitors[ligne].judges[juge][vague - 1] = {
            note: note,
            coul: entry.coul
        };

        // Store the note for average calculation if no interference
        if (!competitors[ligne].averages[vague - 1]) {
            competitors[ligne].averages[vague - 1] = [];
        }
        competitors[ligne].averages[vague - 1].push(note);
    });

    // Calculate the best averages and the total
    Object.keys(competitors).forEach(ligne => {
        let validAverages = competitors[ligne].averages.map((notes, index) => {
            if (competitors[ligne].interferenceFlags[index] || notes.length < 3) {
                // Skip calculation if interference is detected or there are fewer than 3 notes
                return '-';
            }
            let sortedNotes = notes.sort((a, b) => a - b);
            if (sortedNotes.length > 3) {
                sortedNotes = sortedNotes.slice(1, - 1); // Remove the smallest and largest note
            }
            const avg = (sortedNotes.reduce((a, b) => a + b, 0) / sortedNotes.length).toFixed(2);
            return parseFloat(avg);
        }).filter(avg => avg !== '-'); // Filter out invalid averages

        // Sort the averages to find the two best
        validAverages.sort((a, b) => b - a);
        competitors[ligne].bestTwo[0] = validAverages[0] || 0;
        competitors[ligne].bestTwo[1] = validAverages[1] || 0;

        // Adjust the total and second-best score based on priority or interference
        if (competitors[ligne].hasPriority) {
            // Priority case: second-best score is 0
            competitors[ligne].total = competitors[ligne].bestTwo[0];
        } else if (competitors[ligne].interferenceCount > 0) {
            // Interference case: divide second-best score by 2
            let adjustedSecondBest = competitors[ligne].bestTwo[1] / (2 ** competitors[ligne].interferenceCount);
            competitors[ligne].total = (competitors[ligne].bestTwo[0] + adjustedSecondBest).toFixed(2);
        } else {
            // Normal case: sum of best two scores
            competitors[ligne].total = (competitors[ligne].bestTwo[0] + competitors[ligne].bestTwo[1]).toFixed(2);
        }
    });

    return competitors;
}




function renderAverageRow(averages, bestTwo, maxVagues, serie, concu, interferenceFlags, hasPriority, interferenceCount) {
    let averageRowHtml = '<tr class="average-row"><td><strong>Moyenne</strong></td>';

    let calculatedAverages = averages.map((notes, waveIndex) => {
        // Check if there's an interference flag for this wave
        if (interferenceFlags && interferenceFlags[waveIndex]) {
            return '▲'; // Display the triangle symbol for interference
        }

        if (!notes || notes.length === 0) return '-';
        let sortedNotes = [...notes].sort((a, b) => a - b);
        if (sortedNotes.length > 3) {
            sortedNotes = sortedNotes.slice(1, -1); // Remove smallest and largest
        }
        const avg = (sortedNotes.reduce((a, b) => a + b, 0) / sortedNotes.length).toFixed(2);
        return parseFloat(avg);
    });

    // Track if we have already highlighted the best two scores
    let bestTwoHighlighted = { best: false, second: false };

    // Display the averages with color, highlighting best scores in red and bold
    calculatedAverages.forEach((avg, waveIndex) => {
        let highlight = '';

        if (!bestTwoHighlighted.best && avg === bestTwo[0]) {
            highlight = 'style="color: red; font-weight: bold;"';
            bestTwoHighlighted.best = true;
        } else if (!bestTwoHighlighted.second && avg === bestTwo[1]) {
            highlight = 'style="color: red; font-weight: bold;"';
            bestTwoHighlighted.second = true;
        }

        // Check if priority exists for this competitor (second best is 0)
        if (hasPriority && avg === bestTwo[1]) {
            avg = 0;
        }

        // Check if interference exists (second best is divided by 2)
        if (interferenceCount === 1 && avg === bestTwo[1]) {
            avg = (avg / 2).toFixed(2); // Division by 2 and round to 2 decimal places
        } else if (interferenceCount === 2 && avg === bestTwo[1]) {
            avg = 0; // Completely remove second best if there are 2 interferences
        }

        // Add a class 'average-cell' to these specific cells, and include data-serie, data-wave, and data-concu attributes
        averageRowHtml += `<td class="average-cell" ${highlight} data-serie="${serie}" data-wave="${waveIndex + 1}" data-concu="${concu}">${avg}</td>`;
    });

    // Fill remaining empty columns up to 15 if necessary
    for (let i = calculatedAverages.length; i < maxVagues; i++) {
        averageRowHtml += `<td>-</td>`;
    }

    averageRowHtml += '</tr>';
    return averageRowHtml;
}







// Function to fetch data and update the page content
    function fetchAndUpdateContent() {
        // Fetch data from PHP script
        fetch('fetch_data.php')
    .then(response => response.json())
    .then(response => {
        const maxVagues = 15;
        const data = response.data;
        //console.log("Données reçues:", data); // Ajoutez ceci pour déboguer
        const headerData = response.header;
        const tableData = prepareTableData(data, maxVagues);
        renderHeader(headerData);
        renderCompetitorInfo(tableData);
        renderScoreTable(tableData, maxVagues);
    })
    .catch(error => console.error('Error fetching data:', error));

    }




function renderCompetitorInfo(competitors) {
    const competitorInfoSection = document.querySelector('#competitor-info');
    competitorInfoSection.style.width = "60%"; // Limit the width to 60%

    let competitorInfoHtml = '<table><thead><tr><th>Rank</th><th>Name</th><th>Club</th><th>Best Score 1</th><th>Best Score 2</th><th>Total</th><th>To Beat 1st</th><th>To Beat 2nd</th></tr></thead><tbody>';
    let competitorList = [];
    let sortedByTotal = [];

    // Add all competitors to the list and sort by total to determine ranking
    Object.keys(competitors).forEach(ligne => {
        competitorList.push(competitors[ligne]);
    });

    // Sort by total to determine rank
    sortedByTotal = [...competitorList].sort((a, b) => b.total - a.total);

    const firstPlaceTotal = sortedByTotal[0].total;
    const secondPlaceTotal = sortedByTotal.length > 1 ? sortedByTotal[1].total : 0;

    // Render each competitor in order of rank
    sortedByTotal.forEach(competitor => {
        let rank = sortedByTotal.indexOf(competitor) + 1;
        let pointsToBeatFirst = '', pointsToBeatSecond = '';

        // Handle priority and interference logic for the header
        let displaySecondBest = competitor.bestTwo[1]; // Default to the actual second best
        if (competitor.hasPriority) {
            // If the competitor has priority, second best is displayed as 0
            displaySecondBest = 0;
        } else if (competitor.interferenceCount > 0) {
            // If the competitor has interference(s), reduce second best accordingly
            if (competitor.interferenceCount === 1) {
                displaySecondBest = (competitor.bestTwo[1] / 2).toFixed(2); // Ensure rounding
            } else if (competitor.interferenceCount === 2) {
                displaySecondBest = 0; // Remove the second best completely if there are 2 interferences
            }
        }

        // Determine the competitor target and calculate the "needs" value
        if (rank === 1) {
            pointsToBeatFirst = 'Win';
            pointsToBeatSecond = 'Win';
        } else if (rank === 2) {
            // For the 2nd place, calculate only to beat 1st
            pointsToBeatFirst = calculatePointsNeededWithInterference(
                competitor, firstPlaceTotal
            );
        } else {
            // For other competitors, calculate to beat 1st and to beat 2nd
            pointsToBeatFirst = calculatePointsNeededWithInterference(
                competitor, firstPlaceTotal
            );
            pointsToBeatSecond = calculatePointsNeededWithInterference(
                competitor, secondPlaceTotal
            );
        }

        // Apply the competitor's color (coul) to the row
        const colorClass = competitor.judges[Object.keys(competitor.judges)[0]][0].coul; // Get the color class

        // Add competitor info to the table with the new order of columns, including interference/priority handling
        competitorInfoHtml += `
            <tr class="${colorClass}">
                <td>#${rank}</td>
                <td>${competitor.nom}</td>
                <td>${competitor.club}</td>
                <td>${competitor.bestTwo[0].toFixed(2)}</td> <!-- Round best score 1 -->
                <td>${displaySecondBest}</td>
                <td>${competitor.total}</td>
                <td>${pointsToBeatFirst}</td>
                <td>${pointsToBeatSecond}</td>
            </tr>`;
    });

    competitorInfoHtml += '</tbody></table>';

    // Insert the competitor info HTML into the page
    competitorInfoSection.innerHTML = competitorInfoHtml;
}




function calculatePointsNeeded(competitor, targetTotal) {
    const bestScore = competitor.bestTwo[0];
    let secondBest = competitor.bestTwo[1]; // Default second best score

    // Adjust second best based on interference count
    if (competitor.hasPriority) {
        secondBest = 0;  // Ignore second best if priority exists
    } else if (competitor.interferenceCount === 1) {
        secondBest = secondBest / 2; // Halve the second best score
    } else if (competitor.interferenceCount === 2) {
        secondBest = 0; // Set second best to 0 if two interferences exist
    }

    // Case 1: Try replacing the second-best score
    let scoreToBeat = (parseFloat(targetTotal) + 0.01 - bestScore).toFixed(2);
    if (scoreToBeat <= 10 && scoreToBeat > secondBest) {
        return parseFloat(scoreToBeat);
    }

    // Case 2: Try replacing the best score, consider the second-best in total
    scoreToBeat = (parseFloat(targetTotal) + 0.01 - secondBest).toFixed(2);
    if (scoreToBeat <= 10) {
        return parseFloat(scoreToBeat);
    }

    // Case 3: Cannot beat the target with a maximum of 10 points
    return parseFloat(targetTotal + 0.01);
}

function calculatePointsNeededWithInterference(competitor, targetTotal) {
    targetTotal = parseFloat(targetTotal) || 0;
 console.log('Competitor data:', competitor);
    console.log('Target total:', targetTotal);
    let bestScore = parseFloat(competitor.bestTwo[0]) || 0;
    let secondBestScore = parseFloat(competitor.bestTwo[1]) || 0;
    let maxScore = 10; // Maximum possible score for a wave

    // Case: Priority or Double Interference (second best is treated as 0)
    if (competitor.hasPriority || competitor.interferenceCount === 2) {
        secondBestScore = 0; // Second best is set to 0

        // Calculate if they can beat the target with just the best score
        let currentTotal = bestScore; // Only the best score is considered

        if (targetTotal <= maxScore) {
            // If the difference between targetTotal and currentTotal is less than or equal to maxScore
            return parseFloat((targetTotal + 0.01));
        } else {
            // If not, it's impossible to beat the target
            return "impossible";
        }
    }

    // Case: Single Interference (second best is halved)
  if (competitor.interferenceCount === 1) {
    secondBestScore = secondBestScore / 2;

    let X = (targetTotal - bestScore) * 2;
  

    if (X > 10) {
        if (X > targetTotal) {
            if (X > 20) {
                return "impossible"; 
            } else if ((targetTotal - (bestScore / 2)) < 10) {
        console.log(targetTotal - (bestScore / 2));
        return (targetTotal - (bestScore / 2)).toFixed(2);
    } else {
        return "impossible"; 
    }
        } else {
            return (X + 0.01).toFixed(2);  // Ensure X is converted to a fixed decimal string
        }
    } else if ((targetTotal - (bestScore / 2)) < 10) {
        
        return (targetTotal - (bestScore / 2)).toFixed(2);
    } else {
        return (targetTotal + 0.01).toFixed(2);
    }
}


    // Case 1: Replacing second best (if it's enough to beat the target)
    if (bestScore + maxScore >= targetTotal) {
        let required = parseFloat((targetTotal - bestScore + 0.01).toFixed(2));
        return Math.min(required, maxScore); // Cap the need by the max possible score (10)
    }

    // Case 2: Replacing the best score to beat the target
    if (maxScore + secondBestScore >= targetTotal) {
        let required = parseFloat((targetTotal - secondBestScore + 0.01).toFixed(2));
        return Math.min(required, maxScore); // Cap the need by the max possible score (10)
    }

    // Case 3: No possible combination to beat the target
    return parseFloat((targetTotal + 0.01).toFixed(2));
}


function renderScoreTable(competitors, maxVague = 15) {
    const tableBody = document.querySelector('#scores-table tbody');
    const tableHead = document.querySelector('#scores-table thead');
    let tableHtml = '';
    const maxVagues = 15;
    
    // Generate the wave headers (W1, W2, W3...) with exactly 15 columns
    let waveHeaderHtml = '<tr><th></th>'; // First empty cell for judge column
    for (let i = 1; i <= maxVagues; i++) {
        waveHeaderHtml += `<th>W${i}</th>`;
    }
    waveHeaderHtml += '</tr>';

    // Insert the wave header row into the table header
    tableHead.innerHTML = waveHeaderHtml;

    // Render each competitor's scores
    Object.keys(competitors).forEach(ligne => {
        let competitor = competitors[ligne];

        // Render each judge's scores with color applied to empty cells
        Object.keys(competitor.judges).forEach(judge => {
            let rowHtml = `<tr><td>J${judge}</td>`; // First column for the judge
            competitor.judges[judge].forEach((wave, index) => {
                const colorClass = wave.coul || `coul-${competitor.ligne}`; // Get the color class or use default color

                // Ensure serie and concu are included
                const serie = competitor.serie;
                const concu = competitor.concu;

                if (wave.note) {
                    // Make the cell editable and include serie and concu
                    rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${index + 1}">${wave.note}</td>`;
                } else {
                    rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${index + 1}">-</td>`; // Editable empty cells as well
                }
            });

            // Fill remaining empty wave columns up to 15 if needed
            for (let i = competitor.judges[judge].length; i < maxVagues; i++) {
                const colorClass = `coul-${competitor.ligne}`;
                const serie = competitor.serie;
                const concu = competitor.concu;

                rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${i + 1}">-</td>`;
            }

            rowHtml += '</tr>';
            tableHtml += rowHtml;
        });

        // Render averages row with best scores highlighted in red and bold
        // Now pass interferenceCount and hasPriority to renderAverageRow
        tableHtml += renderAverageRow(
            competitor.averages,
            competitor.bestTwo,
            maxVagues,
            competitor.serie,
            competitor.concu,
            competitor.interferenceFlags,
            competitor.hasPriority,
            competitor.interferenceCount
        );
    });

    // Insert the table HTML into the page
    tableBody.innerHTML = tableHtml;

    // Add event listener for editing notes (if needed)
    addEditableListeners();
}







function addEditableListeners() {
    document.querySelectorAll('td[contenteditable="true"]').forEach(cell => {
        // Event to save on "Enter" key press or blur event
        cell.addEventListener('keydown', function (event) {
            if (event.key === 'Enter') {
                event.preventDefault();
                cell.blur();  // Trigger blur to save the content
            }
        });
        
        cell.addEventListener('focus', function() {
             // Arrête le rafraîchissement automatique
            
            clearInterval(autoRefreshInterval);
            autoRefreshInterval = null;
            });
        
        // Event to handle when the user leaves the cell (blur)
        cell.addEventListener('blur', function () {
            const newValue = cell.textContent.trim();  // Get the new value from the cell
            const serie = cell.getAttribute('data-serie');
            const concu = cell.getAttribute('data-concu');
            const judge = cell.getAttribute('data-judge');
            const wave = cell.getAttribute('data-wave');

            // Debug pour vérifier les valeurs
            console.log(`Serie: ${serie}, Concu: ${concu}, Judge: ${judge}, Wave: ${wave}, Note: ${newValue}`);
            
            
            // Si la note est valide, effectuer une requête INSERT ou UPDATE
            if (!isNaN(newValue)) {
                if (newValue === "") {
                    updateNoteInDatabase(serie, concu, judge, wave, 0);
                
                //if (cell.textContent === "-" || cell.textContent === "") {
                    // Si la cellule était vide, effectuer une requête INSERT
                //    insertNoteInDatabase(serie, concu, judge, wave, newValue);
                } else {
                    // Si la cellule avait déjà une note, effectuer une requête UPDATE
                    updateNoteInDatabase(serie, concu, judge, wave, newValue);
                }
            } else {
                alert("Veuillez entrer une note valide.");
                // Option pour réinitialiser la cellule si la note n'est pas valide
                cell.textContent = "-";
            }
            autoRefreshInterval = setInterval(fetchAndUpdateContent, 2000);
        });
    });
    
}

function insertNoteInDatabase(serie, concu, judge, wave, note) {
    // Créer les données à envoyer
    const data = {
        serie: serie,
        concu: concu,
        judge: judge,
        wave: wave,
        note: note
    };
    console.log("Données envoyées pour l'insertion:", data);

    // Envoyer la requête au serveur pour insérer la nouvelle note dans la base de données
    fetch('insert_note.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        if (!response.ok) {
            return response.text().then(text => { throw new Error(text); });
        }
        return response.json();
    })
    .then(result => {
        if (result.status === 'success') {
            console.log('Note insérée avec succès.');
        } else {
            console.error('Erreur lors de l\'insertion de la note:', result.message);
            alert("Erreur lors de l'insertion de la note.");
        }
    })
    .catch(error => {
        console.error('Erreur:', error.message);
        alert("Erreur lors de l'insertion : " + error.message);
    });
}



// Function to update the note in the database via fetch (AJAX)
function updateNoteInDatabase(serie, concu, judge, wave, note) {
    // Créer les données à envoyer
    const data = {
        serie: serie,
        concu: concu,
        judge: judge,
        wave: wave,
        note: note
    };
 //console.log("Données envoyées:", data);
    // Envoyer la requête au serveur pour mettre à jour la note dans la base de données
    fetch('update_note.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        // Lire la réponse et vérifier si elle est en JSON
        if (!response.ok) {
            return response.text().then(text => { throw new Error(text); });
        }
        return response.json();
    })
    .then(result => {
        if (result.status === 'success') {
            //console.log('Note mise à jour avec succès.');
        } else {
            console.error('Erreur lors de la mise à jour de la note:', result.message);
            //alert("Erreur lors de la mise à jour de la note.");
        }
    })
    .catch(error => {
        console.error('Erreur:', error.message);
        //alert("Erreur lors de la mise à jour : " + error.message);
    });
}









