/* script_controler.js — version réorganisée, sans doublons, même API */

/* ========= ÉTAT GLOBAL ========= */
const state = {
  autoRefreshInterval: null,
  countdownInterval: null,
  selectedHeat: null,
  globalserie: null,
  mode: null,
  nbrejuges: null,
  refreshvalue: null,
  pause: null,
  isAutoEdit: false,
  isAutoValide: false
};

/* ========= HELPERS GÉNÉRIQUES ========= */
function stopAutoRefresh() {
  if (state.autoRefreshInterval) {
    clearInterval(state.autoRefreshInterval);
    state.autoRefreshInterval = null;
  }
}

function clearCountdownInterval() {
  if (state.countdownInterval) {
    clearInterval(state.countdownInterval);
    state.countdownInterval = null;
  }
}

/* ========= CHRONO ========= */
function updateRemainingTime(targetTimeSRC) {
  clearCountdownInterval();
  state.countdownInterval = setInterval(function () {
    const targetTime = new Date(targetTimeSRC);
    const now = new Date();
    const diff = targetTime - now;

    if (diff <= 0) {
      document.getElementById('remaining-time').innerText = '00:00';
      clearCountdownInterval();
      return;
    }

    const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
    const seconds = Math.floor((diff % (1000 * 60)) / 1000);
    const formattedTime =
      (minutes < 10 ? '0' : '') + minutes + ':' + (seconds < 10 ? '0' : '') + seconds;

    document.getElementById('remaining-time').innerText = formattedTime;
  }, 1000);
}

function startChrono(targetTime = null) {
  if (!targetTime) {
    const minutes = parseInt(document.getElementById('minutes').value, 10);
    state.nbrejuges = parseInt(document.getElementById('judges').value, 10);
    fetch(`start_chrono.php?minutes=${minutes}&judges=${state.nbrejuges}`)
      .then(r => r.json())
      .then(data => {
        if (data.status === 'success') {
          clearCountdownInterval();
          updateRemainingTime(data.targettime);
        } else {
          console.error('Error starting chrono:', data.message);
        }
      })
      .catch(err => console.error('Error starting chrono:', err));
  }
}

function playChrono() {
  fetch(`play_chrono.php`)
    .then(r => r.json())
    .then(data => {
      if (data.status === 'success') {
        clearCountdownInterval();
        updateRemainingTime(data.targettime);
      } else {
        console.error('Error starting chrono:', data.message);
      }
    })
    .catch(err => console.error('Error starting chrono:', err));
}

function stopChrono() {
  fetch('stop_chrono.php')
    .then(r => r.json())
    .then(data => {
      if (data.status === 'success') {
        clearCountdownInterval();
      } else {
        console.error('Error starting chrono:', data.message);
      }
    })
    .catch(err => console.error('Error starting chrono:', err));

  clearCountdownInterval();
  document.getElementById('remaining-time').innerText = '00:00';
}

function pauseChrono() {
  const chrono_remaining = document.getElementById('remaining-time').innerText;
  fetch(`pause_chrono.php?remaining=${encodeURIComponent(chrono_remaining)}`)
    .then(r => r.json())
    .then(data => {
      if (data.status === 'success') {
        clearCountdownInterval();
      } else {
        console.error('Error pausing chrono:', data.message);
      }
    })
    .catch(err => console.error('Error pausing chrono:', err));
  clearCountdownInterval();
}

/* ========= LECTURE INFOS INIT ========= */
function loadbloqueFromDatabase() {
  fetch('getbloque.php')
    .then(r => r.json())
    .then(data => {
      if (Array.isArray(data) && data.length > 0) {
        const retrievedValue = data[0].serie;
        if (parseInt(data[0].nbrjug, 10) > 0) {
          state.nbrejuges = parseInt(data[0].nbrjug, 10);
        } else {
          state.nbrejuges = parseInt(data[0].valeur, 10);
        }
        document.getElementById('minutes').value = data[1]?.valeur ?? '';
        document.getElementById('judges').value = state.nbrejuges;

        state.globalserie = retrievedValue;
        state.mode = 'live';
        fetchAndUpdateContent(state.globalserie);
      } else {
        console.error('Aucune donnée trouvée dans getbloque.php');
      }
    })
    .catch(err => console.error('Erreur lors de la récupération des données:', err));
}

/* ========= RENDER HEADER ========= */
function renderHeader(headerData) {
  document.getElementById('division-info').textContent = headerData.division;
  document.getElementById('round-info').textContent = headerData.tour;
  document.getElementById('heat-info').textContent = headerData.heat;
  if (headerData.nbrjug > 0) {
    state.nbrejuges = headerData.nbrjug;
    document.getElementById('judges').value = state.nbrejuges;
  }
}

/* ========= TABLE: PREPARE LOGIC ========= */
function prepareTableData(data, maxVagues) {
  const competitors = {};
  const isAutoValide = state.isAutoValide;
  const nbrejuges = state.nbrejuges;

  data.forEach(entry => {
    const ligne = parseInt(entry.ligne, 10);
    const juge = parseInt(entry.juge, 10);
    const vague = parseInt(entry.vague, 10);
    const concu = parseInt(entry.concu, 10);
    const note = parseFloat(entry.note);
    const prio = parseInt(entry.prio, 10);

    let validevalue = isAutoValide ? 1 : 0;

    if (!competitors[ligne]) {
      competitors[ligne] = {
        nom: entry.nom,
        club: entry.club,
        ligne,
        serie: entry.serie,
        concu,
        judges: {},
        averages: [],
        bestTwo: [0, 0],
        moyennes: [],
        total: 0,
        interferenceFlags: Array(maxVagues).fill(false),
        valideFlags: Array(maxVagues).fill(validevalue),
        hasPriority: false,
        interferenceCount: 0,
        coul: entry.coul,
        rank: 0,
        prio: prio,
        needs: ''
      };
    }

    if (isAutoValide) {
      competitors[ligne].valideFlags[vague - 1] = 1;
    } else {
      competitors[ligne].valideFlags[vague - 1] = parseInt(entry.valide, 10);
    }

    // Interférences/priorité (juge = 0 avec note 11 ou 14)
    if (juge === 0 && (note === 11 || note === 14)) {
      competitors[ligne].interferenceFlags[vague - 1] = true;
      if (note === 14) competitors[ligne].hasPriority = true;
      if (note === 11) competitors[ligne].interferenceCount++;
      return;
    }

    if (!competitors[ligne].judges[juge]) {
      competitors[ligne].judges[juge] = Array(maxVagues).fill('-');
    }

    competitors[ligne].judges[juge][vague - 1] = {
      note: note,
      coul: entry.coul
    };

    if (!competitors[ligne].averages[vague - 1]) {
      competitors[ligne].averages[vague - 1] = [];
    }
    competitors[ligne].averages[vague - 1].push(note);
  });

  // Calcul moyennes, bestTwo, total, classement, needs
  Object.keys(competitors).forEach(ligne => {
    const c = competitors[ligne];

    let validAverages = c.averages
      .map((notes, index) => {
        let sortedNotes = (notes || []).slice().sort((a, b) => a - b);
        let avg = 0;

        if (sortedNotes.length >= nbrejuges) {
          if (nbrejuges > 3) sortedNotes = sortedNotes.slice(1, -1);
          avg = (sortedNotes.reduce((a, b) => a + b, 0) / sortedNotes.length).toFixed(2);
        } else if (sortedNotes.length > 0) {
          avg = 21; // signal "incomplet"
        } else {
          avg = 0;
        }

        if (c.interferenceFlags[index]) avg = 11;
        if (c.valideFlags[index] !== 1) avg = 'NV';

        c.moyennes.push({
          id: index,
          value: parseFloat(avg),
          isBestScore: false,
          color: 'gray',
          nbrejuge: nbrejuges,
          nbrenotes: sortedNotes.length
        });

        return parseFloat(avg);
      })
      .filter(avg => avg !== '-' && avg !== 21 && avg !== 11 && avg !== 'NV');

    validAverages.sort((a, b) => b - a);
    c.bestTwo[0] = validAverages[0] || 0;
    c.bestTwo[1] = validAverages[1] || 0;

    c.moyennes.sort((a, b) => b.value - a.value);

    let bestScoreCount = 0;
    c.moyennes.forEach(m => {
      if (m.value < 11 && bestScoreCount < 2 && (m.value === c.bestTwo[0])) {
        m.isBestScore = true;
        m.color = 'red';
        bestScoreCount++;
      } else if (m.value < 11 && bestScoreCount < 2 && (m.value === c.bestTwo[1])) {
        if (c.hasPriority) {
          m.value = 0;
        } else if (c.interferenceCount > 0) {
          m.value = c.bestTwo[1] / (2 ** c.interferenceCount);
        }
        m.isBestScore = true;
        m.color = 'red';
        bestScoreCount++;
      }
    });

    if (c.hasPriority) {
      c.total = c.bestTwo[0];
    } else if (c.interferenceCount > 0) {
      const adjusted = c.bestTwo[1] / (2 ** c.interferenceCount);
      c.total = (c.bestTwo[0] + adjusted).toFixed(2);
    } else {
      c.total = (c.bestTwo[0] + c.bestTwo[1]).toFixed(2);
    }

    c.moyennes.sort((a, b) => a.id - b.id);
  });

  // Classement + needs
  const sorted = Object.values(competitors).sort((a, b) => {
    if (b.total !== a.total) return b.total - a.total;
    const A = [...a.moyennes].sort((x, y) => y.value - x.value);
    const B = [...b.moyennes].sort((x, y) => y.value - x.value);
    for (let i = 0; i < Math.min(A.length, B.length); i++) {
      if (B[i].value !== A[i].value) return B[i].value - A[i].value;
    }
    return 0;
  });

  const first = sorted[0]?.total ?? 0;
  const second = sorted[1]?.total ?? 0;

  sorted.forEach((c, i) => {
    c.rank = i + 1;
    if (c.rank === 1) {
      c.needs = 'Win';
    } else if (c.rank === 2) {
      c.needs = calculatePointsNeededWithInterference(c, first);
    } else {
      c.needs =
        calculatePointsNeededWithInterference(c, first) + ' (1st), ' +
        calculatePointsNeededWithInterference(c, second) + ' (2nd)';
    }
  });

  return competitors;
}

/* ========= TABLE: RENDER ========= */
function renderAverageRow(averages, bestTwo, maxVagues, serie, concu, interferenceFlags, hasPriority, interferenceCount, valideFlags) {
  let html = '<tr class="average-row"><td><strong>Moyenne</strong></td>';
  const nbrejuges = state.nbrejuges;

  const calculated = averages.map((notes, waveIndex) => {
    if (interferenceFlags && interferenceFlags[waveIndex]) return '▲';
    if (!notes || notes.length === 0) return '-';
    if (notes.length > 0 && notes.length < nbrejuges) return '-';

    let sorted = [...notes].sort((a, b) => a - b);
    if (sorted.length > 3) sorted = sorted.slice(1, -1);
    const avg = (sorted.reduce((a, b) => a + b, 0) / sorted.length).toFixed(2);
    return parseFloat(avg);
  });

  const highlighted = { best: false, second: false };

  calculated.forEach((avg, idx) => {
    let highlight = '';
    if (!highlighted.best && avg === bestTwo[0]) {
      highlight = 'style="color: red; font-weight: bold;"';
      highlighted.best = true;
    } else if (!highlighted.second && avg === bestTwo[1]) {
      highlight = 'style="color: red; font-weight: bold;"';
      highlighted.second = true;
    }

    if (hasPriority && avg === bestTwo[1]) {
      avg = 0;
    }

    if (interferenceCount === 1 && avg === bestTwo[1]) {
      avg = (avg / 2).toFixed(2);
    } else if (interferenceCount === 2 && avg === bestTwo[1]) {
      avg = 0;
    }

    if (valideFlags[idx] === 1) {
      html += `<td class="average-cell" ${highlight} data-serie="${serie}" data-wave="${idx + 1}" data-concu="${concu}">${avg}</td>`;
    } else {
      html += `<td style="background-color: gray; color: lightgray;" class="average-cell" ${highlight} data-serie="${serie}" data-wave="${idx + 1}" data-concu="${concu}">${avg}</td>`;
    }
  });

  for (let i = calculated.length; i < maxVagues; i++) html += `<td>-</td>`;
  html += '</tr>';
  return html;
}

function renderScoreTable(competitors, maxVague = 15) {
  const tableBody = document.querySelector('#scores-table tbody');
  const tableHead = document.querySelector('#scores-table thead');
  let tableHtml = '';
  const maxVagues = 15;

  // En-tête W1..W15
  let waveHeaderHtml = '<tr><th></th>';
  for (let i = 1; i <= maxVagues; i++) waveHeaderHtml += `<th>W${i}</th>`;
  waveHeaderHtml += '</tr>';
  tableHead.innerHTML = waveHeaderHtml;

  Object.keys(competitors).forEach(ligne => {
    const competitor = competitors[ligne];

    Object.keys(competitor.judges).forEach(judge => {
      let rowHtml = `<tr><td>J${judge}</td>`;
      competitor.judges[judge].forEach((wave, index) => {
        const colorClass = wave.coul || `coul-${competitor.ligne}`;
        const serie = competitor.serie;
        const concu = competitor.concu;

        if (wave.note) {
          rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${index + 1}">${wave.note}</td>`;
        } else {
          rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${index + 1}">-</td>`;
        }
      });

      for (let i = competitor.judges[judge].length; i < maxVagues; i++) {
        const colorClass = `coul-${competitor.ligne}`;
        const serie = competitor.serie;
        const concu = competitor.concu;
        rowHtml += `<td class="${colorClass}" contenteditable="true" data-serie="${serie}" data-concu="${concu}" data-judge="${judge}" data-wave="${i + 1}">-</td>`;
      }

      rowHtml += '</tr>';
      tableHtml += rowHtml;
    });

    tableHtml += renderAverageRow(
      competitor.averages,
      competitor.bestTwo,
      maxVagues,
      competitor.serie,
      competitor.concu,
      competitor.interferenceFlags,
      competitor.hasPriority,
      competitor.interferenceCount,
      competitor.valideFlags
    );
  });

  tableBody.innerHTML = tableHtml;
  addEditableListeners();
}

/* ========= PANEL DROITE / RANKING ========= */
function renderCompetitorInfo(competitors) {
  const section = document.querySelector('#competitor-info');
  section.style.width = '60%';

  let html = '<table style="width:100%;"><thead><tr><th>Rank</th><th>Name</th><th>Club</th><th>Total</th><th>To 1st</th><th>To 2nd</th></tr></thead><tbody>';

  const list = Object.keys(competitors).map(k => competitors[k]);

  const sorted = [...list].sort((a, b) => {
    if (b.total !== a.total) return b.total - a.total;
    const A = [...a.moyennes].sort((x, y) => y.value - x.value);
    const B = [...b.moyennes].sort((x, y) => y.value - x.value);
    for (let i = 0; i < Math.min(A.length, B.length); i++) {
      if (B[i].value !== A[i].value) return B[i].value - A[i].value;
    }
    return 0;
  });

  const first = sorted[0]?.total ?? 0;
  const second = sorted[1]?.total ?? 0;

  sorted.forEach(competitor => {
    const rank = sorted.indexOf(competitor) + 1;
    let pointsToBeatFirst = '';
    let pointsToBeatSecond = '';

    let displaySecondBest = competitor.bestTwo[1].toFixed(2);
    if (competitor.hasPriority) displaySecondBest = 0;
    else if (competitor.interferenceCount > 0) {
      displaySecondBest =
        (competitor.interferenceCount === 1)
          ? (competitor.bestTwo[1] / 2).toFixed(2)
          : 0;
    }

    if (rank === 1) {
      pointsToBeatFirst = 'Win';
      pointsToBeatSecond = 'Win';
    } else if (rank === 2) {
      pointsToBeatFirst = calculatePointsNeededWithInterference(competitor, first);
    } else {
      pointsToBeatFirst = calculatePointsNeededWithInterference(competitor, first);
      pointsToBeatSecond = calculatePointsNeededWithInterference(competitor, second);
    }

    if (parseFloat(competitor.total) > 0) {
      updateScores(state.globalserie, competitor.ligne, rank, competitor.total);
    } else {
      updateScores(state.globalserie, competitor.ligne, null, null);
    }

    const colorClass = competitor.coul;
    html += `
      <tr class="${colorClass}">
        <td>#${rank}</td>
        <td>${competitor.nom}</td>
        <td>${competitor.club}</td>
        <td>
          <div style="display:flex;flex-direction:column;align-items:center;">
            <div style="text-align:right;font-weight:bold;">${competitor.total}</div>
            <div>${competitor.bestTwo[0].toFixed(2)} - ${displaySecondBest}</div>
          </div>
        </td>
        <td style="vertical-align:bottom;text-align:right;">${pointsToBeatFirst}</td>
        <td style="vertical-align:bottom;text-align:right;">${pointsToBeatSecond}</td>
      </tr>`;
  });

  html += '</tbody></table>';
  section.innerHTML = html;

  // Priorités
  const priorities = [];
  Object.keys(competitors).forEach(ligne => {
    const c = competitors[ligne];
    if (c.prio && c.prio !== 0) priorities.push({ ligne: c.ligne, prio: c.prio, coul: c.coul });
  });
  priorities.sort((a, b) => a.prio - b.prio);

  let pHtml = '<table style="width:100%;"><tbody><tr>';
  priorities.forEach(c => { pHtml += `<td style="font-weight:bold;text-align:center;" class="${c.coul}">P${c.prio}</td>`; });
  pHtml += '</tr></tbody></table>';

  document.querySelector('#prioritytable').innerHTML = pHtml;
}

/* ========= FETCH & BIND ========= */
function fetchAndUpdateContent(serie) {
  state.globalserie = serie;
  fetch(`fetch_data.php?serie=${encodeURIComponent(serie)}`)
    .then(r => r.json())
    .then(response => {
      const maxVagues = 15;
      const data = response.data;
      const headerData = response.header;
      const tableData = prepareTableData(data, maxVagues);

      renderHeader(headerData);
      renderCompetitorInfo(tableData);
      renderScoreTable(tableData, maxVagues);
      bindContextMenu();

      if (headerData && headerData.chrono_state === 'running') {
        updateRemainingTime(headerData.targettime);
      } else if (headerData && headerData.chrono_state === 'paused') {
        document.getElementById('remaining-time').innerText = headerData.chrono_remaining;
      } else {
        document.getElementById('remaining-time').innerText = '00:00';
        updateRemainingTime('1970-01-01T00:00:00Z');
      }

      prepareAndSendTableData(tableData, headerData);
    })
    .catch(err => console.error('Error fetching data:', err));
}

function prepareAndSendTableData(tableData, headerData) {
  const output = {
    compete: state.globalserie,
    categ: headerData.division,
    round: headerData.tour,
    heat: headerData.heat,
    chrono: headerData.targettime,
    chrono_state: headerData.chrono_state,
    chrono_remaining: headerData.chrono_remaining,
    competitors: []
  };

  Object.keys(tableData).forEach(ligne => {
    const c = tableData[ligne];
    const competitorData = {
      name: c.nom,
      score: c.total,
      needs: c.needs,
      position: c.rank,
      Competitor_Id: ligne,
      club: c.club,
      interf: c.interferenceCount,
      ligne: c.ligne,
      lycra: c.coul,
      scores: [],
      moyennes: []
    };

    c.averages.forEach((scoresArray, vagueIndex) => {
      scoresArray.forEach(score => {
        competitorData.scores.push({
          id: vagueIndex + 1,
          value: score,
          color: ''
        });
      });
    });

    c.moyennes.forEach((m, idx) => {
      if (c.valideFlags[idx] === 1) {
        competitorData.moyennes.push({
          value: m.value,
          isBestScore: m.isBestScore,
          id: idx + 1,
          color: m.color
        });
      }
    });

    output.competitors.push(competitorData);
  });

  const jsonToSend = [output];
  const autovalide = state.isAutoValide ? 1 : 0;

  fetch(`../savedata.php?mode=${encodeURIComponent(state.mode ?? '')}&autovalide=${autovalide}`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(jsonToSend)
  })
    .then(r => r.text())
    .then(data => {
      console.log('Success payload:', jsonToSend);
      console.log('Success response:', data);
      state.mode = '';
    })
    .catch(err => console.error('Error saving data:', err));
}

/* ========= CALCUL NEEDS ========= */
function calculatePointsNeeded(competitor, targetTotal) {
  const bestScore = competitor.bestTwo[0];
  let secondBest = competitor.bestTwo[1];

  if (competitor.hasPriority) secondBest = 0;
  else if (competitor.interferenceCount === 1) secondBest = secondBest / 2;
  else if (competitor.interferenceCount === 2) secondBest = 0;

  let scoreToBeat = (parseFloat(targetTotal) + 0.01 - bestScore).toFixed(2);
  if (scoreToBeat <= 10 && scoreToBeat > secondBest) return parseFloat(scoreToBeat);

  scoreToBeat = (parseFloat(targetTotal) + 0.01 - secondBest).toFixed(2);
  if (scoreToBeat <= 10) return parseFloat(scoreToBeat);

  return parseFloat((targetTotal + 0.01).toFixed(2));
}

function calculatePointsNeededWithInterference(competitor, targetTotal) {
  targetTotal = parseFloat(targetTotal) || 0;
  let bestScore = parseFloat(competitor.bestTwo[0]) || 0;
  let secondBestScore = parseFloat(competitor.bestTwo[1]) || 0;
  const maxScore = 10;

  if (competitor.hasPriority || competitor.interferenceCount === 2) {
    secondBestScore = 0;
    if (targetTotal <= maxScore) {
      return parseFloat((targetTotal + 0.01));
    }
    return 'impossible';
  }

  if (competitor.interferenceCount === 1) {
    secondBestScore = secondBestScore / 2;
    let X = (targetTotal - bestScore) * 2;

    if (X > 10) {
      if (X > targetTotal) {
        if (X > 20) {
          return 'impossible';
        } else if ((targetTotal - (bestScore / 2)) < 10) {
          return (targetTotal - (bestScore / 2)).toFixed(2);
        } else {
          return 'impossible';
        }
      } else {
        return (X + 0.01).toFixed(2);
      }
    } else if ((targetTotal - (bestScore / 2)) < 10) {
      return (targetTotal - (bestScore / 2)).toFixed(2);
    } else {
      return (targetTotal + 0.01).toFixed(2);
    }
  }

  if (bestScore + maxScore >= targetTotal) {
    const required = parseFloat((targetTotal - bestScore + 0.01).toFixed(2));
    return Math.min(required, maxScore);
  }

  if (maxScore + secondBestScore >= targetTotal) {
    const required = parseFloat((targetTotal - secondBestScore + 0.01).toFixed(2));
    return Math.min(required, maxScore);
  }

  return parseFloat((targetTotal + 0.01).toFixed(2));
}

/* ========= DB UPDATES ========= */
function insertNoteInDatabase(serie, concu, judge, wave, note) {
  const data = { serie, concu, judge, wave, note };
  fetch('insert_note.php', {
    method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data)
  })
    .then(r => { if (!r.ok) return r.text().then(t => { throw new Error(t); }); return r.json(); })
    .then(res => { if (res.status !== 'success') throw new Error(res.message); })
    .catch(err => { console.error('Insertion error:', err.message); alert("Erreur lors de l'insertion : " + err.message); });
}

function updateNoteInDatabase(serie, concu, judge, wave, note) {
  const data = { serie, concu, judge, wave, note };
  fetch('update_note.php', {
    method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data)
  })
    .then(r => { if (!r.ok) return r.text().then(t => { throw new Error(t); }); return r.json(); })
    .then(res => {
      if (res.success !== true) console.error('Erreur update note:', res.message);
    })
    .catch(err => console.error('Erreur update note:', err.message));
}

function updatebloque(serie) {
  const data = { serie };
  fetch('update_bloque.php', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data) })
    .catch(err => console.error('Erreur update_bloque:', err.message));
}

function updateScores(serie, ligne, rank, total) {
  const data = { serie, ligne, rank, total };
  fetch('update_score_total.php', { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data) })
    .catch(err => console.error('Erreur update_score_total:', err.message));
}

/* ========= CONTEXT MENU ========= */
function removeExistingListeners() {
  ['cancel', 'remove-interference', 'interference-in', 'priority-in'].forEach(id => {
    const el = document.getElementById(id);
    if (el) el.replaceWith(el.cloneNode(true));
  });
}

function hideContextMenu() {
  const menu = document.getElementById('context-menu');
  if (menu) menu.style.display = 'none';
}

function updateDatabase(action, serie, concu, wave) {
  const data = { action, serie, concu, wave };
  fetch('update_action.php', {
    method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data)
  })
    .then(r => { if (!r.ok) throw new Error(r.statusText); return r.json(); })
    .then(result => {
      if (result.status !== 'success') console.error('Error:', result.message);
    })
    .catch(err => console.error('Error:', err))
    .finally(() => {
      fetchAndUpdateContent(serie);
      hideContextMenu();
    });
}

function showContextMenu(event, cell) {
  const contextMenu = document.getElementById('context-menu');
  const x = event.pageX;
  const y = event.pageY;

  contextMenu.style.left = `${x}px`;
  contextMenu.style.top = `${y}px`;
  contextMenu.style.display = 'block';

  const serie = cell.getAttribute('data-serie');
  const concu = cell.getAttribute('data-concu');
  const wave = cell.getAttribute('data-wave');

  removeExistingListeners();
  document.getElementById('cancel').addEventListener('click', hideContextMenu);
  document.getElementById('remove-interference').addEventListener('click', () => updateDatabase('REMOVE INTERFERENCE', serie, concu, wave));
  document.getElementById('interference-in').addEventListener('click', () => updateDatabase('INTERFERENCE IN', serie, concu, wave));
  document.getElementById('priority-in').addEventListener('click', () => updateDatabase('PRIORITY IN', serie, concu, wave));
}

function bindContextMenu() {
  document.querySelectorAll('.average-cell').forEach(cell => {
    cell.addEventListener('click', (event) => {
      event.preventDefault();
      showContextMenu(event, cell);
    });
  });
}

document.addEventListener('click', function (event) {
  const contextMenu = document.getElementById('context-menu');
  if (!event.target.closest('.average-cell') && !event.target.closest('#context-menu')) {
    if (contextMenu) contextMenu.style.display = 'none';
  }
});

/* ========= LISTENERS D’ÉDITION ========= */
function addEditableListeners() {
  document.querySelectorAll('td[contenteditable="true"]').forEach(cell => {
    let oldValue = null;

    cell.addEventListener('focus', function () {
      stopAutoRefresh();
      oldValue = cell.textContent.trim();
    });

    cell.addEventListener('keydown', function (event) {
      if (event.key === 'Enter') {
        event.preventDefault();
        let newValue = cell.textContent.trim();

        if (oldValue !== newValue) {
          const serie = cell.getAttribute('data-serie');
          const concu = cell.getAttribute('data-concu');
          const judge = cell.getAttribute('data-judge');
          const wave = cell.getAttribute('data-wave');

          if (!isNaN(newValue) && newValue !== '') {
            newValue = parseFloat(newValue).toFixed(2);
            updateNoteInDatabase(serie, concu, judge, wave, newValue);
            cell.textContent = newValue;
          } else {
            alert('Veuillez entrer une note valide.');
            cell.textContent = '-';
          }
        }

        if (state.isAutoEdit) {
          const nextCell = this.nextElementSibling;
          if (nextCell && nextCell.getAttribute('contenteditable') === 'true') {
            nextCell.focus();
            document.execCommand('selectAll', false, null);
          }
        } else {
          if (state.refreshvalue === true) {
            stopAutoRefresh();
            state.autoRefreshInterval = setInterval(() => fetchAndUpdateContent(state.globalserie), 2000);
          }
          fetchAndUpdateContent(state.globalserie);
        }
      }
    });

    cell.addEventListener('input', function () {
      let newValue = cell.textContent.trim();
      if (!isNaN(newValue) && newValue !== '') {
        if (newValue.match(/^0\d+$/)) {
          newValue = (parseFloat(newValue) / 10).toString();
        } else {
          let parsed = parseFloat(newValue);
          if (parsed > 10 && Number.isInteger(parsed)) {
            newValue = (parsed / 10).toString();
          } else if (parsed <= 10 && newValue.includes('.')) {
            const [i, d] = newValue.split('.');
            newValue = d.length > 2 ? `${i}.${d.slice(0, 2)}` : newValue;
          }
        }
        cell.textContent = newValue;

        const range = document.createRange();
        const sel = window.getSelection();
        range.setStart(cell.childNodes[0], newValue.length);
        range.collapse(true);
        sel.removeAllRanges();
        sel.addRange(range);
      }
    });
  });
}

/* ========= CATEGORIES / ROUNDS / HEATS ========= */
function fetchCategories() {
  const categoriesContainer = document.getElementById('categoriesContainer');
  fetch('getcateg.php')
    .then(r => r.json())
    .then(data => {
      categoriesContainer.innerHTML = '';
      const def = document.createElement('option');
      def.textContent = 'Select a category';
      def.value = '';
      categoriesContainer.appendChild(def);

      data.forEach(category => {
        const opt = document.createElement('option');
        opt.textContent = category.categ;
        opt.value = category.categ;
        categoriesContainer.appendChild(opt);
      });

      categoriesContainer.addEventListener('change', function () {
        if (this.value) fetchRounds(this.value);
      }, { once: true }); // évite multiples attaches
    })
    .catch(err => console.error('Error fetching categories:', err));
}

function fetchRounds(category) {
  const roundsContainer = document.getElementById('roundsContainer');
  fetch(`getrounds.php?category=${encodeURIComponent(category)}`)
    .then(r => r.json())
    .then(data => {
      roundsContainer.innerHTML = '';
      const def = document.createElement('option');
      def.textContent = 'Select a round';
      def.value = '';
      roundsContainer.appendChild(def);

      data.forEach(round => {
        const opt = document.createElement('option');
        opt.textContent = round.round.trim();
        opt.value = round.round.trim();
        roundsContainer.appendChild(opt);
      });

      roundsContainer.addEventListener('change', function () {
        if (this.value) fetchHeats(category, this.value);
      }, { once: true });
    })
    .catch(err => console.error('Error fetching rounds:', err));
}

function fetchHeats(category, round) {
  const heatsContainer = document.getElementById('heatsContainer');
  fetch(`getheats.php?category=${encodeURIComponent(category)}&round=${encodeURIComponent(round)}`)
    .then(r => r.json())
    .then(data => {
      heatsContainer.innerHTML = '';
      const ul = document.createElement('ul');

      for (const [serie, heats] of Object.entries(data)) {
        for (const [heat] of Object.entries(heats)) {
          const li = document.createElement('li');
          li.textContent = heat;
          li.setAttribute('data-heat', serie);
          li.addEventListener('click', function () {
            state.selectedHeat = this.getAttribute('data-heat');
            state.globalserie = state.selectedHeat;
            state.mode = '';
            fetchAndUpdateContent(state.selectedHeat);
          });
          ul.appendChild(li);
        }
      }
      heatsContainer.appendChild(ul);
    })
    .catch(err => console.error('Error fetching heats:', err));
}

/* ========= INIT DOMCONTENTLOADED ========= */
document.addEventListener('DOMContentLoaded', function () {
  const refreshToggle = document.getElementById('refresh-toggle');
  const manualRefreshBtn = document.getElementById('manual-refresh-btn');
  const startChronoBtn = document.getElementById('start-chrono-btn');
  const stopChronoBtn = document.getElementById('stop-chrono-btn');
  const pauseChronoBtn = document.getElementById('pause-chrono-btn');
  const playChronoBtn = document.getElementById('play-chrono-btn');
  const viewheatbtn = document.getElementById('view-btn');
  const activateheatbtn = document.getElementById('activate-btn');
  const autoEditToggle = document.getElementById('auto-edit-toggle');
  const autoValideToggle = document.getElementById('auto-valide-toggle');

  // État initial
  autoValideToggle.checked = false;
  state.isAutoValide = false;

  loadbloqueFromDatabase();
  fetchCategories();

  // Toggles
  refreshToggle.addEventListener('change', function () {
    if (refreshToggle.checked) {
      state.refreshvalue = true;
      stopAutoRefresh();
      state.autoRefreshInterval = setInterval(() => fetchAndUpdateContent(state.globalserie), 2000);
    } else {
      state.refreshvalue = false;
      stopAutoRefresh();
    }
  });

  manualRefreshBtn.addEventListener('click', function () {
    fetchAndUpdateContent(state.globalserie);
  });

  startChronoBtn.addEventListener('click', startChrono);
  stopChronoBtn.addEventListener('click', stopChrono);
  playChronoBtn.addEventListener('click', playChrono);
  pauseChronoBtn.addEventListener('click', pauseChrono);

  viewheatbtn.addEventListener('click', function () {
    if (state.selectedHeat) {
      window.location.href = `../livescoreshisto.php?heatNumber=${encodeURIComponent(state.selectedHeat)}`;
    }
  });

  activateheatbtn.addEventListener('click', function () {
    updatebloque(state.globalserie);
    state.mode = 'live';
  });

  autoEditToggle.addEventListener('change', function () {
    state.isAutoEdit = this.checked;
  });

  autoValideToggle.addEventListener('change', function () {
    state.isAutoValide = this.checked;
  });
});
