<?php
/**
 * Script PHP pour extraire la "meilleure ligne" par compétiteur,
 * en construisant dynamiquement le mapping des rounds à partir de la table "heats"
 * via la requête : SELECT DISTINCT round FROM heats ORDER BY heat
 * et en exportant le résultat vers un CSV.
 */

 // Configuration de la base de données
// Database configuration
require 'config_livescores2024.php';


// 2) Charger la correspondance des points depuis la table "points"
$pointsMapping = [];
$sqlPoints = "SELECT * FROM points";
$resultPoints = $conn->query($sqlPoints);
if ($resultPoints && $resultPoints->num_rows > 0) {
    while ($row = $resultPoints->fetch_assoc()) {
        // On suppose que le champ "rang" contient le classement et "points" la valeur correspondante
        $pointsMapping[(int)$row["rang"]] = $row["points"];
    }
}
$resultPoints->free();

// 3) Récupérer les informations de la compétition depuis la table "compet"
$sqlCompet = "SELECT intitule, lieu FROM compet LIMIT 1";
$resultCompet = $conn->query($sqlCompet);
$competInfo = $resultCompet->fetch_assoc();
$resultCompet->free();

$intitule = isset($competInfo['intitule']) ? $competInfo['intitule'] : 'competition';
$lieu     = isset($competInfo['lieu']) ? $competInfo['lieu'] : '';

// Définir le nom du fichier CSV en fonction de l'intitulé (remplacement des espaces par des underscores)
$filename = "classement_" . preg_replace('/\s+/', '_', $intitule) . ".csv";

// 4) Construction dynamique du mapping des rounds via la requête : SELECT DISTINCT round FROM heats ORDER BY heat
$roundMapping = [];
$sqlDistinct = "SELECT DISTINCT round FROM heats ORDER BY heat";
$resultDistinct = $conn->query($sqlDistinct);
$priorityCounter = 1;
if ($resultDistinct && $resultDistinct->num_rows > 0) {
    while ($row = $resultDistinct->fetch_assoc()) {
        $roundValue = $row['round'];
        // Affecter une priorité selon l'ordre renvoyé par la requête (1 pour le premier, 2 pour le deuxième, etc.)
        $roundMapping[$roundValue] = $priorityCounter;
        $priorityCounter++;
    }
}
$resultDistinct->free();

// 5) Récupération de toutes les lignes de la table "heats"
$sql = "SELECT * FROM heats";
$result = $conn->query($sql);
$data = [];
if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        // Ajout de la priorité du round à partir du mapping
        $round = $row["round"];
        $rp = isset($roundMapping[$round]) ? $roundMapping[$round] : 0;
        $row["round_priority"] = $rp;
        
        // Conversion de la colonne 'rank' en nombre.
        // Si la valeur est vide ou non numérique, on utilise 9999 pour la placer en fin de tri.
        $rank_value = 9999;
        if (!empty($row["rank"]) && is_numeric($row["rank"])) {
            $rank_value = (float)$row["rank"];
        }
        $row["rank_numeric"] = $rank_value;
        
        $data[] = $row;
    }
}
$result->free();

// 6) Regrouper les lignes par compétiteur (champ "concu")
$by_competitor = [];
foreach ($data as $row) {
    $concu = $row["concu"];
    if (!isset($by_competitor[$concu])) {
        $by_competitor[$concu] = [];
    }
    $by_competitor[$concu][] = $row;
}

// 7) Pour chaque compétiteur, sélectionner la "meilleure ligne"
//    On trie chaque groupe par : round_priority décroissant, puis par rank_numeric croissant
$best_rows = [];
foreach ($by_competitor as $concu => $rows) {
    usort($rows, function($a, $b) {
        $diff_rp = $b["round_priority"] - $a["round_priority"];
        if ($diff_rp !== 0) {
            return $diff_rp;
        }
        if ($a["rank_numeric"] == $b["rank_numeric"]) {
            return 0;
        }
        return ($a["rank_numeric"] < $b["rank_numeric"]) ? -1 : 1;
    });
    // La première ligne du tableau trié est la "meilleure" pour ce compétiteur
    $best_rows[] = $rows[0];
}

// 8) Regrouper les "meilleures lignes" par catégorie (champ "categ")
$grouped_by_categ = [];
foreach ($best_rows as $row) {
    $categ = $row["categ"];
    if (!isset($grouped_by_categ[$categ])) {
        $grouped_by_categ[$categ] = [];
    }
    $grouped_by_categ[$categ][] = $row;
}

// 9) Pour chaque catégorie, trier les enregistrements et calculer la position
//    en appliquant un classement de type "competition ranking"
//    (les compétiteurs ayant le même couple (round_priority, rank_numeric) obtiennent la même position)
foreach ($grouped_by_categ as $cat => &$rows) {
    usort($rows, function($a, $b) {
        $diff = $b["round_priority"] - $a["round_priority"];
        if ($diff !== 0) {
            return $diff;
        }
        if ($a["rank_numeric"] == $b["rank_numeric"]) {
            return 0;
        }
        return ($a["rank_numeric"] < $b["rank_numeric"]) ? -1 : 1;
    });
    $lastPos = 0;
    foreach ($rows as $i => &$row) {
        if ($i === 0) {
            $row["position"] = 1;
            $lastPos = 1;
        } else {
            // Si le couple (round_priority, rank_numeric) est identique à la ligne précédente, même position
            if ($row["round_priority"] == $rows[$i-1]["round_priority"] && $row["rank_numeric"] == $rows[$i-1]["rank_numeric"]) {
                $row["position"] = $lastPos;
            } else {
                $row["position"] = $i + 1;
                $lastPos = $row["position"];
            }
        }
    }
    unset($row);
}
unset($rows);

// 10) Fusionner les groupes par catégorie pour créer un tableau final
$final_rows = [];
ksort($grouped_by_categ); // Trie par ordre alphabétique des catégories
foreach ($grouped_by_categ as $cat => $rows) {
    foreach ($rows as $row) {
        $final_rows[] = $row;
    }
}

// 11) Pour chaque ligne, ajouter la colonne "points" en se basant sur la correspondance de la table "points"
foreach ($final_rows as &$row) {
    $pos = $row["position"];
    $row["points"] = isset($pointsMapping[$pos]) ? $pointsMapping[$pos] : "";
}
unset($row);

// 12) Exporter le CSV avec en-tête personnalisé et forcer le téléchargement
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="'.$filename.'"');

$output = fopen('php://output', 'w');

// Écriture de l'en-tête personnalisé avec l'intitulé et le lieu de la compétition
fputcsv($output, ["Intitulé", $intitule], ";");
fputcsv($output, ["Lieu", $lieu], ";");
fputcsv($output, []); // Ligne vide

// En-tête du tableau
fputcsv($output, ["position", "categ", "concu", "nom", "club", "round", "rank", "points"], ";");

// Écriture des lignes finales dans le CSV
foreach ($final_rows as $row) {
    
    $competStmt = $conn->prepare("UPDATE concu SET rank1= ?, points= ? WHERE numpart= ? ");
    $competStmt->bind_param("iii", $row["position"], $row["points"] ,$row["concu"] );
    $competStmt->execute();
    
    fputcsv($output, [
        $row["position"],
        $row["categ"],
        $row["concu"],
        $row["nom"],
        $row["club"],
        $row["round"],
        $row["rank"],
        $row["points"]
    ], ";");
}

fclose($output);
$conn->close();
exit;
?>